#ifndef __ISCSI_LICENSE_STRUCT_H__
#define __ISCSI_LICENSE_STRUCT_H__

/* current version 1.4 */
#define L_VERSION_MAJOR		1
#define L_VERSION_MINOR		4
/* L_VERSION_MAJOR/MINOR_MIN is the minimum version that uses current license
 * 	structure (i.e., size, fields ...)
 * We may bump up the version when a new value is added for a existing field
 * But these 2 defines will be unchanged as long as the structure fields and
 * 	size remain the same
 * They are used in the license file so the key generated by a newer version
 * of the chkeygentool can be used by an older driver 
  */
#define L_VERSION_MAJOR_MIN	1 
#define L_VERSION_MINOR_MIN	3 
#define VERSION_STR_LENGTH	8 /* 4 digits each for ver major and minor */

#define ETH_MAC_ADDR_MAX_LEN	6 /* mac address: 6 bytes */

#define SN_MAX_LEN	16
struct eth_info {
	unsigned char	mac[ETH_MAC_ADDR_MAX_LEN];  /* MAC address */
	unsigned short  linkspeed;
	unsigned short  vendor_id;
	unsigned short  device_id;
	unsigned char	deviceType[SN_MAX_LEN];
	unsigned char	serialNum[SN_MAX_LEN];
	unsigned short  reeval;
};

/*
 * customer info file format:
 * fields:
 *	@eths:		4 digits, hex, # of interfaces found in the system,
 *			record max. of INFOFILE_ETH_MAX.
 *	@eth_selected:	4 digits, hex, the index of the interface user selected
 *			(0-based)
 *	@eth information:
 *		- struct eth_info
 *		- VPD
 *
 * fields at the end of the file:
 *	@version:	
 *		- major,	4 digits
 *		- minor,	4 digits
 */

#define ETH_VPD_BYTES 	8

/* max. # of interfaces to be scanned, during customer info. gathering */
#define INFOFILE_ETH_MAX	32

#define INFOFILE_DELIMIT_MAX	6
#define INFOFILE_DELIMITER	0xFF

/* infofile data includes
 * - ethmax,
 * - selected eth idx,
 * - INFOFILE_ETH_MAX of infofile_eth_info, and
 * - INFOFILE_DELIMIT_MAX of INFOFILE_DELIMITER at the end
 */
#define INFOFILE_DATA_BYTES	\
	(sizeof(unsigned short) + /* ethmax */ \
	sizeof(unsigned short) + /* selected eth idx */ \
	INFOFILE_ETH_MAX * \
	(sizeof(struct eth_info) + ETH_VPD_BYTES) /* eth info */ + \
	INFOFILE_DELIMIT_MAX)

#define INFOFILE_DATA_STEP(infofilesize)	\
	((infofilesize) / (INFOFILE_DATA_BYTES))
#define INFOFILE_DATA_START(infofilesize)	\
        ((infofilesize) % (INFOFILE_DATA_BYTES))

static inline unsigned int infofile_size(void)
{
        /* 
	 * calculate the min. infofile size to accommendate INFOFILE_DATA_BYTES
         * - each data byte is converted to 2 characters
         * - the size needs to be power of 2
         */

	int bit = 31;
	unsigned int mask = (1 << bit);
	unsigned int val = INFOFILE_DATA_BYTES;

	val <<= 1; /* bytes to string */

	/* find the msb, if the remaining bits are non-zero, shift left */
	for (; bit >= 0; bit--, mask >>= 1)
		if (mask & val)
			break;
	if ((~mask) & val)
		bit++;
	if (INFOFILE_DATA_STEP(1 << bit) < 2)
		bit++;

	return (1 << bit);
}

/*
 * license key file format:
 *
 * fields:
 *	@version-major:		4 bytes
 *	@version-minor:		4 bytes
 *	@customer name:		32 bytes, padded with NULL
 *	@product[PRODUCT_MAX];	
 *		- flag:
 *			product flag,	4 bytes each, hex
 *		- license duration:
 *			* day,		2 bytes
 *			* month,	2 bytes
 *			* year,		4 bytes
 *		- spec1:
 *			product specific field	8 bytes
 *		- spec2:
 *			product specific field	8 bytes
 *	@eths:	# of eth in the license, 2 bytes
 *	@eth information:
 *		- mac address,	6 * 2 bytes, hex
 *		- vendor id,	4 bytes, hex
 *		- device id,	4 bytes, hex
 *		- speed,	4 bytes, hex
 *		- VPD,		8 * 2 bytes, hex
 *
 * fields at the tail end:
 *	@length:	length of the actual license key, 4 bytes
 *
 * fields at the end of the file:
 *	@version:	
 *		- major,	4 bytes
 *		- minor,	4 bytes
 *
 *	@check-sum		8 bytes
 */

/* for license key file */
#define MATRIX_ORDER		128
#define MATRIX_SIZE		(MATRIX_ORDER * MATRIX_ORDER)
#define MATRIX_ORDER_HALF	(MATRIX_ORDER >> 1)
#define CRC_STR_LENGTH		8	/* uint32_t */
#define	KEYFILE_VERSION_OFFSET	(MATRIX_SIZE)
#define	KEYFILE_CRC_OFFSET	(MATRIX_SIZE + VERSION_STR_LENGTH)
#define KEYFILE_LENGTH		(MATRIX_SIZE + VERSION_STR_LENGTH + CRC_STR_LENGTH) 

#define KEYFILE_ETH_MAX		128

/* invalid characters are checked during license key generation */
#define CUSTOMER_NAME_INVALID_CHARS	"`/@!#$%,;^&*()+~|\"\'?><+={}[]:\\"
#define LICENSE_CUSTOMER_NAME_MAX_LEN	31
#define EVAL_DAYS_MAX		248	/* 8 month */

/*
 *			!!!WARNING!!!

 * if you are adding, removing, or changing the order of enum prodcuts
 * please
 *	1. increase the version number. And
 *	2. update license_version_product_max()
 */

enum products {
	PROD_ISCSI_TARGET,
	PROD_ISCSI_INITIATOR,
	PROD_UNIFIED_STORAGE,
	PROD_UNIFIED_GATEWAY,
	PROD_CHIMNEY,
	PROD_UNIFIED_MANAGER,
	PROD_UM_WEBGUI,
	PROD_AMSTERDAM,
	PROD_FCOE,
	PROD_RESERVED1 = PROD_UM_WEBGUI,
	PROD_RESERVED2 = PROD_AMSTERDAM,
	PROD_RESERVED3 = PROD_FCOE,
	PROD_RESERVED4,
	
	PRODUCT_MAX,
	PROCUCT_MAX_VERSION_1_1 = PROD_CHIMNEY,
	PROCUCT_MAX_VERSION_1_2 = PROD_RESERVED3, 
	PRODUCT_MAX_VERSION_1_3 = PRODUCT_MAX, 
	PRODUCT_MAX_VERSION_1_4 = PRODUCT_MAX
};

static inline char *product_string(int product)
{
	switch (product) {
	case PROD_ISCSI_TARGET:		return "iSCSI_Target";
	case PROD_ISCSI_INITIATOR:	return "iSCSI_Initiator";
	case PROD_UNIFIED_STORAGE:	return "Unified_Storage";
	case PROD_UNIFIED_GATEWAY:	return "Unified_Gateway";
	case PROD_CHIMNEY:		return "Chimney";
	case PROD_UNIFIED_MANAGER:	return "Unified Manager";
	case PROD_UM_WEBGUI:		return "Unified Web Management";
	case PROD_AMSTERDAM:		return "Amsterdam";
	case PROD_FCOE:			return "FCoE";
	case PROD_RESERVED4:		return NULL;
	}
	return NULL;
}

enum uss_subtype {
	USS_SUBTYPE_FC,
	USS_SUBTYPE_FCOE,
	USS_SUBTYPE_pNFS,
	USS_SUBTYPE_NAS,
	USS_SUBTYPE_NWRAID,
	USS_SUBTYPE_TP,
	USS_SUBTYPE_IOCACHE,
	USS_SUBTYPE_REPLIC,
	USS_SUBTYPE_RDMA,
	USS_SUBTYPE_HA,
	USS_SUBTYPE_COMPRESS,
	USS_SUBTYPE_DEDUPE,
	USS_SUBTYPE_SNAP,

	USS_SUBTYPE_MAX
};

static inline char* uss_subtype_str(int subtype)
{
	switch(subtype){
	case USS_SUBTYPE_FC:	return "FC";
	case USS_SUBTYPE_FCOE:	return "FCOE_Target";
	case USS_SUBTYPE_pNFS:	return "pNFS";
	case USS_SUBTYPE_NAS:	return "NAS";
	case USS_SUBTYPE_NWRAID:return "NW_RAID";	      
	case USS_SUBTYPE_TP:	return "TP";
	case USS_SUBTYPE_IOCACHE: return "IOCACHE";
	case USS_SUBTYPE_REPLIC:return "Rep";
	case USS_SUBTYPE_RDMA:	return "RDMA";
	case USS_SUBTYPE_HA:	return "HA";
	case USS_SUBTYPE_COMPRESS:return "Compression";
	case USS_SUBTYPE_DEDUPE:return "Dedupe";
	case USS_SUBTYPE_SNAP:	return "CS";
	}
	return NULL;
}

static inline int license_is_usr_subtype(int subtype)
{
	switch(subtype){
	case USS_SUBTYPE_FC:
	case USS_SUBTYPE_FCOE:
	case USS_SUBTYPE_HA:
		return 1;
	}
	return 0;
}

static inline int license_version_product_max(int ver_major, int ver_minor)
{
	if (ver_major == 1 && ver_minor == 1)
		return PROCUCT_MAX_VERSION_1_1;
	if (ver_major == 1 && ver_minor == 2)
		return PROCUCT_MAX_VERSION_1_2;
	if (ver_major == 1 && ver_minor == 3)
		return PRODUCT_MAX_VERSION_1_3;
	if (ver_major == 1 && ver_minor == 4)
		return PRODUCT_MAX_VERSION_1_4;
	return -1;
}

static inline int license_print_product_max(int ver_major, int ver_minor)
{
	if (ver_major == 1 && ver_minor == 1)
		return PROCUCT_MAX_VERSION_1_1;
	if (ver_major == 1 && ver_minor == 2)
		return PROCUCT_MAX_VERSION_1_2;
	if (ver_major == 1 && ver_minor == 3)
		return PROD_RESERVED3; 
	if (ver_major == 1 && ver_minor == 4)
		return PROD_RESERVED4; 
	return -1;
}

enum flag_common_bits {
	FLAG_EVAL_BIT,
	FLAG_PROD_BIT,
	FLAG_UNIVERSAL_BIT,
	FLAG_USS_MAINT_BIT,
	FLAG_COMMON_BITS
};

static inline char *license_flag_bit_string(int bit)
{
	switch (bit) {
	case FLAG_EVAL_BIT:	return "EVAL";
	case FLAG_PROD_BIT:	return "PROD";
	case FLAG_UNIVERSAL_BIT: return "UNIVS";
	case FLAG_USS_MAINT_BIT: return "USS_MAINT";
	}
	return NULL;
}

struct key_duration {
	unsigned int	day;
	unsigned int	mon;
	unsigned int	year;
};
#define EVAL_KEY_DURATION_DAY_DFLT	31
#define PROD_KEY_DURATION_YRS_DFLT	25
#define USS_MAINT_KEY_DURATION_YRS_DFLT	1

#define USS_SIZE_DFLT	32
struct license {
	unsigned short		version[2];	/* version major, minor */
	char			customer[LICENSE_CUSTOMER_NAME_MAX_LEN + 1];
	struct{
		unsigned short	flag;
		struct key_duration	duration;

		unsigned int	spec1;
		unsigned int	spec2;
	}product[PRODUCT_MAX];
	unsigned short		ethmax;
	struct eth_info 	eth[0];
};

static inline int key_duration_invalid(struct key_duration *duration)
{
	return (duration->day > 31) ||( duration->mon > 12);
}

static inline void license_set_flag(struct license *license, int prod, int bit)
{
	license->product[prod].flag |= 1 << bit;
}

static inline void license_clear_flag(struct license *license, int prod, int bit)
{
	license->product[prod].flag &= ~(1 << bit);
}
static inline unsigned int license_flag_set(struct license *license, int prod,
					    int bit)
{
	return license->product[prod].flag & (1 << bit);
}

static inline void license_set_duration(struct license *license, int prod,
					int day, int mon, int yr)
{
	license->product[prod].duration.day = day;
	license->product[prod].duration.mon = mon;
	license->product[prod].duration.year = yr;
}

static inline void license_set_size(struct license *license, int prod, int size)
{
	license->product[PROD_UNIFIED_STORAGE].spec1 = size; 
}

static inline unsigned int
license_get_size(struct license *license, int prod)
{
	return license->product[PROD_UNIFIED_STORAGE].spec1; 
}

static inline void license_set_uss_subtype(struct license *license, int type)
{
	license->product[PROD_UNIFIED_STORAGE].spec2 |= (1<<type);
}

static inline int license_get_uss_subtype(struct license *license, int type)
{
	return license->product[PROD_UNIFIED_STORAGE].spec2 & (1<<type);
}

static inline void license_set_usr_subtype(struct license *license, int type)
{
	license->product[PROD_UNIFIED_GATEWAY].spec2 |= (1<<type);
}

static inline int license_get_usr_subtype(struct license *license, int type)
{
	return license->product[PROD_UNIFIED_GATEWAY].spec2 & (1<<type);
}

static inline int license_get_amsterdam_subtype(struct license *license, int type)
{
	return license->product[PROD_AMSTERDAM].spec2 & (1<<type);
}

static inline void license_set_amsterdam_subtype(struct license *license, int type)
{
	license->product[PROD_AMSTERDAM].spec2 |= (1<<type);
}

/*
 * permutation & reserse permutation
 */
static inline int midx(int order, int row, int col)
{
	return row * order + col;
}

static inline void buf_to_matrix(int order, char *buf, char *matrix)
{
	int i, j, k;

	for (i = 0, k = 0; i < order; i++)
		for (j = 0; j < order; j++, k++)
			matrix[midx(order,i,j)] = buf[k];
}

static inline void matrix_to_buf(int order, char *buf, char *matrix)
{
	int i, j, k;

	for (i = 0, k = 0; i < order; i++)
		for (j = 0; j < order; j++, k++)
			buf[k] = matrix[midx(order,i,j)];
}

#ifndef __KLIB__
static inline void permutate_1(int order, char *f_matrix, char *t_matrix)
{
	int i, j, ti, tj;
	for (j = 0, ti = 0; j < order; j += 2, ti++) {
                for (i = 0; i < order; i++) {
                        tj = i;
                        t_matrix[midx(order,ti,tj)] = f_matrix[midx(order,i,j)];
                }
        }
}

static inline void permutate_2(int order, char *f_matrix, char *t_matrix)
{
	int i, j, ti, tj;
	
	for (j = 1, ti = order >> 1; j < order;
	     j += 2, ti++) {
                for (i = 0; i < order; i++) {
                        tj = i;
                        t_matrix[midx(order,ti,tj)] = f_matrix[midx(order,i,j)];
                }
        }
}

static inline void permutate_3(int order, char *f_matrix, char *t_matrix)
{
	int i, j;

	for (i = 0; i < order; i++)
                for (j = 0; j < order; j++)
			t_matrix[midx(order, i,order - j - 1)] =
						f_matrix[midx(order,i,j)];
}

static inline void encrypt_key_buffer(int order, char *buf, char *matrix,
				char *pmatrix)
{
	buf_to_matrix(order, buf, matrix);
	permutate_1(order, matrix, pmatrix);
	permutate_2(order, matrix, pmatrix);
	matrix_to_buf(order, buf, pmatrix);

	buf_to_matrix(order, buf, matrix);
	permutate_3(order, matrix, pmatrix);
	matrix_to_buf(order, buf, pmatrix);
}
#endif

static inline void reverse_permutate_1(int order, char *f_matrix, char *t_matrix)
{
	int i, j, ti, tj;
	int half_order = order / 2;
	
	for (i = half_order, tj = 1; i < order;
	     i++, tj += 2)
		for (j = 0; j < order; j++) {
			ti = j;
                        t_matrix[midx(order,ti,tj)] = f_matrix[midx(order,i,j)];
		}
}

static inline void reverse_permutate_2(int order, char *f_matrix, char *t_matrix)
{
	int i, j, ti, tj;
	int half_order = order / 2;

	for (i = 0; i < half_order; i++)
		for (j = 0; j < order; j++) {
			ti = j;
			tj = i << 1;
                        t_matrix[midx(order,ti,tj)] = f_matrix[midx(order,i,j)];
		}
}

static inline void reverse_permutate_3(int order, char *f_matrix, char *t_matrix)
{
	int i, j;

	for (i = 0; i < order; i++)
                for (j = 0; j < order; j++)
			t_matrix[midx(order, i,order - j - 1)] =
						f_matrix[midx(order,i,j)];
}

/* reverse the permutation algorithm */
static inline void decrypt_key_buffer(int order, char *buf, char *matrix,
					char *rmatrix)
{
	buf_to_matrix(order, buf, matrix);
	reverse_permutate_3(order, matrix, rmatrix);
	matrix_to_buf(order, buf, rmatrix);

	buf_to_matrix(order, buf, matrix);
	reverse_permutate_2(order, matrix, rmatrix);
	reverse_permutate_1(order, matrix, rmatrix);

	matrix_to_buf(order, buf, rmatrix);
}

#endif
